/*
 * OgreBulletExample.cpp
 *
 *  Created on: Feb 3, 2011
 *      Author: cpresser
 */

#include "OgreBulletExample.h"
#include <OgreBulletCollisionsShape.h>
#include <Shapes/OgreBulletCollisionsStaticPlaneShape.h>
#include <Shapes/OgreBulletCollisionsBoxShape.h>
#include <Shapes/OgreBulletCollisionsSphereShape.h>
#include <Shapes/OgreBulletCollisionsGImpactShape.h>
#include <Shapes/OgreBulletCollisionsTrimeshShape.h>
#include <Shapes/OgreBulletCollisionsCompoundShape.h>
#include <Utils/OgreBulletCollisionsMeshToShapeConverter.h>

#include "DotSceneLoader.h"

OgreBulletExample::OgreBulletExample():
	m_numEntities(0),
	m_moveSpeed(50),
	m_gravity(0, -9.81, 0),
	m_bounds(),
	m_cubeSpeed(30),
	m_direction(0,0,0),
	m_startMoving(false),
	objShape(0)

{

}

OgreBulletExample::~OgreBulletExample() {
	//empty the queues, free memory
	std::deque<OgreBulletDynamics::RigidBody *>::iterator itBody = m_bodies.begin();
	while(m_bodies.end() != itBody){
		delete *itBody;
		++itBody;
	}

	std::deque<OgreBulletCollisions::CollisionShape *>::iterator itShape = m_shapes.begin();
	while(m_shapes.end() != itShape){
		delete *itShape;
		++itShape;
	}

	m_bodies.clear();
	m_shapes.clear();

	delete m_world;

}

void OgreBulletExample::createScene(void)
{


	////////////////////////////////////////////////////////////////////////////
	//set up OgreBullet
	m_world = new OgreBulletDynamics::DynamicsWorld(mSceneMgr, m_bounds, m_gravity);
	m_debugDrawer = new OgreBulletCollisions::DebugDrawer();
	m_debugDrawer->setDrawWireframe(true);

	m_world->setDebugDrawer(m_debugDrawer);
	m_world->setShowDebugShapes(true);

	//load blender scene (cross fingers)
	Ogre::DotSceneLoader dsl;
	dsl.parseDotScene("blockScene.xml", "General", mSceneMgr);


	//print out dynamic objects
	std::cout << "Dynamic Objects" << std::endl;
	for(Ogre::uint32 i = 0; i < dsl.dynamicObjects.size(); i++){
		std::cout << "\t" << dsl.dynamicObjects[i] << std::endl;
	}

	//print out static object
	std::cout << "Static Objects" << std::endl;
	for(Ogre::uint32 i = 0; i < dsl.staticObjects.size(); i++){
		std::cout << "\t" << dsl.staticObjects[i] << std::endl;
	}

	//print out node properties
	std::cout << "Node Properties" << std::endl;
	for(Ogre::uint32 i = 0; i < dsl.nodeProperties.size(); i++){
		std::cout << "\t" << dsl.nodeProperties[i].nodeName << ", "
				<< "\t" << dsl.nodeProperties[i].propertyNm << ", "
				<< "\t" << dsl.nodeProperties[i].typeName << ", "
				<< "\t" << dsl.nodeProperties[i].valueName
				<< std::endl;
	}

	//make RigidBodies out of dynamics objects

	for(Ogre::uint32 i = 0; i < dsl.dynamicObjects.size(); i++){

		//get the entity
		Ogre::Entity* entity = mSceneMgr->getEntity(dsl.dynamicObjects[i]);

		//get its scene node
		Ogre::SceneNode* node = entity->getParentSceneNode();

		//make a collision shape???
		//we could base it by the entities name (e.g. Cube... turns into a box shape)
		//or we could just decompose everything

		OgreBulletCollisions::CollisionShape *shape = 0;

		//make a rigid body
		OgreBulletDynamics::RigidBody *body = new  OgreBulletDynamics::RigidBody(
				dsl.dynamicObjects[i] + "_body",
				m_world);


		if(dsl.dynamicObjects[i].find("Plane") != Ogre::String::npos){
			//it is a plane
			Ogre::Vector3 scale = node->getScale();
			Ogre::Vector3 bounds = entity->getBoundingBox().getSize();
			bounds = (0.98* bounds * scale)/2;
			std::cout << bounds << std::endl;
			bounds.y = 0.01;
			shape = new OgreBulletCollisions::BoxCollisionShape(bounds);
			body->setStaticShape(node, shape, 0.6f, 0.6f, node->_getDerivedPosition(), node->_getDerivedOrientation());
		}
		else if(dsl.dynamicObjects[i].find("Cube") != Ogre::String::npos){
			//it is a box
			Ogre::Vector3 scale = node->getScale();
			Ogre::Vector3 bounds = entity->getBoundingBox().getSize();
			bounds = (0.98* bounds * scale)/2;
			std::cout << bounds << std::endl;
			shape = new OgreBulletCollisions::BoxCollisionShape(bounds);
			body->setShape(node, shape, 0.6f, 0.6f, 1.0f, node->_getDerivedPosition(), node->_getDerivedOrientation());
			//body->setStaticShape(node, shape, 0.6f, 0.6f, node->_getDerivedPosition(), node->_getDerivedOrientation());
		}
		else if(dsl.dynamicObjects[i].find("Sphere") != Ogre::String::npos){
			//it is a sphere
			Ogre::Vector3 scale = node->getScale();
			Ogre::Vector3 bounds = entity->getBoundingBox().getSize();
			bounds = (0.98* bounds * scale)/2;
			std::cout << bounds << std::endl;
			shape = new OgreBulletCollisions::SphereCollisionShape(bounds.y);
			body->setShape(node, shape, 0.6f, 0.6f, 1.0f, node->_getDerivedPosition(), node->_getDerivedOrientation());
			//body->setStaticShape(node, shape, 0.6f, 0.6f, node->_getDerivedPosition(), node->_getDerivedOrientation());
		}
		else {
			OgreBulletCollisions::StaticMeshToShapeConverter makeShape(entity);
			shape = makeShape.createConvexDecomposition();
			body->setShape(node, shape, 0.6f, 0.6f, 1.0f, node->_getDerivedPosition(), node->_getDerivedOrientation());

			//body->setStaticShape(node, shape, 0.6f, 0.6f, node->_getDerivedPosition(), node->_getDerivedOrientation());


		}

		m_shapes.push_back(shape);
		m_bodies.push_back(body);
	}

}





bool OgreBulletExample::keyPressed(const OIS::KeyEvent & arg)
{
	bool result = BaseApplication::keyPressed(arg);

	switch(arg.key){
	case OIS::KC_B:
		//fire a block
		fireBlock();
		break;
	case OIS::KC_I:
		m_startMoving = true;
		m_direction.z = -m_cubeSpeed;
		break;
	case OIS::KC_K:
		m_startMoving = true;
		m_direction.z = m_cubeSpeed;
		break;
	case OIS::KC_J:
		m_startMoving = true;
		m_direction.x = -m_cubeSpeed;
		break;
	case OIS::KC_L:
		m_startMoving = true;
		m_direction.x = m_cubeSpeed;
		break;
	case OIS::KC_U:
		m_startMoving = true;
		m_direction.y = -m_cubeSpeed;
		break;
	case OIS::KC_O:
		m_startMoving = true;
		m_direction.y = m_cubeSpeed;
		break;
	default:
		break;
	}
//std::cerr << m_direction << std::endl;
	return result;
}

bool OgreBulletExample::keyReleased(const OIS::KeyEvent & arg)
{
	bool result = BaseApplication::keyReleased(arg);

	switch(arg.key){
	case OIS::KC_I:
		m_startMoving = false;
		m_direction.z = 0;
		break;
	case OIS::KC_K:
		m_startMoving = false;
		m_direction.z = 0;
		break;
	case OIS::KC_J:
		m_startMoving = false;
		m_direction.x = 0;
		break;
	case OIS::KC_L:
		m_startMoving = false;
		m_direction.x = 0;
		break;
	case OIS::KC_U:
		m_startMoving = false;
		m_direction.y = 0;
		break;
	case OIS::KC_O:
		m_startMoving = false;
		m_direction.y = 0;
		break;
	default:
		break;
	}
	return result;
}



bool OgreBulletExample::frameEnded(const Ogre::FrameEvent & evt)
{
	bool result = BaseApplication::frameEnded(evt);

	//m_world->stepSimulation(evt.timeSinceLastFrame);

	return result;
}



bool OgreBulletExample::frameStarted(const Ogre::FrameEvent & evt)
{
	bool result = BaseApplication::frameStarted(evt);

	m_world->stepSimulation(evt.timeSinceLastFrame);

	return result;
}



bool OgreBulletExample::frameRenderingQueued(const Ogre::FrameEvent & evt)
{
	bool result = BaseApplication::frameRenderingQueued(evt);


	return result;
}

void OgreBulletExample::fireBlock(){

	//position of new block
	Ogre::Vector3 position = mCamera->getDerivedPosition() +
			mCamera->getDerivedDirection().normalisedCopy() * 10;

	//create an entity
	Ogre::Entity *cubeEntity = mSceneMgr->createEntity("Cube" +
			Ogre::StringConverter::toString(m_numEntities),
			"cube.mesh");

	cubeEntity->setCastShadows(true);

	Ogre::AxisAlignedBox bound = cubeEntity->getBoundingBox();
	Ogre::Vector3 size = bound.getSize();
	size /= 2.0f;
	//size *= 0.95f;

	cubeEntity->setMaterialName("Examples/BumpyMetal");

	//create a node
	Ogre::SceneNode *cubeNode = mSceneMgr->getRootSceneNode()->createChildSceneNode();
	cubeNode->attachObject(cubeEntity);

	//scale both the node and the size
	cubeNode->scale(0.05f, 0.05f, 0.05f);
	size *= 0.05f;

	//create bullet stuff
	OgreBulletCollisions::BoxCollisionShape *cubeShape =
			new OgreBulletCollisions::BoxCollisionShape(size);

	OgreBulletDynamics::RigidBody *cubeBody =
			new OgreBulletDynamics::RigidBody("CubeRigid" +
					Ogre::StringConverter::toString(m_numEntities),
					m_world);

	cubeBody->setShape(cubeNode,
			cubeShape,
			0.6f, //dynamic restitution
			0.6f, //dynamic friction
			1.0f, //dynamic body mass
			position, //starting position
			Ogre::Quaternion(0, 0, 0, 1)  //orientation
			);

	m_numEntities++;

	cubeBody->setLinearVelocity(mCamera->getDerivedDirection().normalisedCopy() * m_moveSpeed);

	m_shapes.push_back(cubeShape);
	m_bodies.push_back(cubeBody);
}

void OgreBulletExample::createConcaveRigidBodyObject(Ogre::String mesh, Ogre::Vector3 position){
	//entity
	Ogre::Entity *objEntity = mSceneMgr->createEntity("Object_" +
			Ogre::StringConverter::toString(m_numEntities),
			mesh);


	//node
	Ogre::SceneNode *objNode = mSceneMgr->getRootSceneNode()->createChildSceneNode();
	objNode->attachObject(objEntity);


	//collision shape
	//OgreBulletCollisions::StaticMeshToShapeConverter makeShape(objEntity);
	//OgreBulletCollisions::CollisionShape *objShape = makeShape.createSphere();
	//OgreBulletCollisions::CollisionShape *objShape = makeShape.createConcave();
	//OgreBulletCollisions::CollisionShape *objShape = makeShape.createConvexDecomposition();

	if(!objShape){
		OgreBulletCollisions::StaticMeshToShapeConverter makeShape(objEntity);
		objShape = makeShape.createConvexDecomposition();
	}
	//rigid body
	OgreBulletDynamics::RigidBody *objBody =
			new OgreBulletDynamics::RigidBody("ObjRigid" +
					Ogre::StringConverter::toString(m_numEntities),
					m_world);

	objBody->setShape(objNode, objShape, 0.6f, 0.6f, 10.0f, position);
	//objBody->setStaticShape(objNode, objShape, 0.6f, 0.6f, position);

	m_numEntities++;

	m_shapes.push_back(objShape);
	m_bodies.push_back(objBody);
}

#if OGRE_PLATFORM == OGRE_PLATFORM_WIN32
#define WIN32_LEAN_AND_MEAN
#include "windows.h"
#endif

#ifdef __cplusplus
extern "C" {
#endif

#if OGRE_PLATFORM == OGRE_PLATFORM_WIN32
    INT WINAPI WinMain( HINSTANCE hInst, HINSTANCE, LPSTR strCmdLine, INT )
#else
    int main(int argc, char *argv[])
#endif
    {
        // Create application object
        OgreBulletExample app;

        try
        {
            app.go();
        } catch( Ogre::Exception& e ) {
#if OGRE_PLATFORM == OGRE_PLATFORM_WIN32
            MessageBox( NULL, e.getFullDescription().c_str(), "An exception has occured!", MB_OK | MB_ICONERROR| MB_TASKMODAL);
#else
            std::cerr << "An exception has occured: " << e.getFullDescription().c_str() << std::endl;
#endif
        }

        return 0;
    }

#ifdef __cplusplus
}


#endif
