/*
 * SimpleGameObject.h
 *
 *  Created on: Mar 18, 2014
 *      Author: cpresser
 */

#ifndef GAMEOBJECT_H_
#define GAMEOBJECT_H_

#include <OgreSceneManager.h>
#include <OgreEntity.h>
#include <OgreFrameListener.h>
#include <OgreString.h>

class GameObject {
public:

	/*
	 * Create an empty uninitialized GameObject
	 *
	 */
	GameObject();

	/*
	 * Create an initialized GameObject
	 *
	 * Parameters:
	 * 	 sceneMgr: the scene manager to add to
	 *   name: the name of the new object
	 *   position: where to place this object
	 *   mesh: the name of the mesh to use with this object
	 *   material: (optional) the name of the material to be used with
	 *   	this object
	 *
	 */
	GameObject(Ogre::SceneManager *sceneMgr,
			Ogre::String name,
			Ogre::Vector3 position,
			Ogre::String mesh,
			Ogre::String material = "",
			Ogre::Vector3 scale = Ogre::Vector3::UNIT_SCALE);

	/*
	 * Destroy the GameObject
	 */
	virtual ~GameObject();

	/**
	 * Update the game object in the scene.
	 * This method will most likely be called when the rest of the
	 * scene is updated e.g. from the frameRenderingQueued method.
	 *
	 * Parameters:
	 *    evt: The frame event sent for the update.
	 */
	virtual void update(const Ogre::FrameEvent& evt);

	/**
	 * Get this objects direction vector as a reference.
	 *
	 * The fact that it returns a reference means that the
	 * method call can be used to change the stored direction.
	 *
	 */
	virtual Ogre::Vector3 &getDirection();

	/**
	 * Set this object's direction.
	 *
	 * Parameter:
	 *    direction: the new direction for this object
	 */
	virtual void setDirection(Ogre::Vector3 direction);

	/**
	 * Set whether or not this object is selected.
	 *
	 * Parameter:
	 *   select: if true select the object, if false deselect it.
	 */
	virtual void setSelected(bool select);

	/**
	 * Return true if this object is selected, false otherwise
	 */
	virtual bool isSelected();

	/**
	 * Return the name of this object.
	 */
	virtual Ogre::String getName();

	/**
	 * Return the name of this object's Entity.
	 */
	virtual Ogre::String getEntityName();

	/**
	 * Return the name of this object's SceneNode.
	 */
	virtual Ogre::String getNodeName();

	/**
	 * Get the scaled bounding box for this object
	 */
	virtual Ogre::AxisAlignedBox getTransformedBoundingBox();

	/**
	 * Move one step back. This will undo a collision.
	 */
	virtual void stepBack();

	/**
	 * Returns true if this object has been initialized.
	 */
	virtual bool isInitialized();


	/**
	 * Initialize the game object so it can be used as part of a scene.
	 * This method will most likely be called when you create a scene.
	 * Only call this method once for an object.
	 *
	 * Parameters:
	 * 	 sceneMgr: the scene manager to add to
	 *   name: the name of the new object
	 *   position: where to place this object
	 *   mesh: the name of the mesh to use with this object
	 *   material: (optional) the name of the material to be used with
	 *   	this object
	 *
	 */
	virtual void init(Ogre::SceneManager *sceneMgr,
			Ogre::String name,
			Ogre::Vector3 position,
			Ogre::String mesh,
			Ogre::String material = "",
			Ogre::Vector3 scale = Ogre::Vector3::UNIT_SCALE);

	int getPoints() const {
		return m_points;
	}

protected:

	/**
	 * A pointer to the SceneManager object to which this object belongs.
	 */
	 Ogre::SceneManager *m_sceneMgr;

	/**
	 * A pointer to the Entity object representing the geometry in the scene graph.
	 */
	Ogre::Entity *m_entity;

	/**
	 * A pointer to the SceneNode object representing the transforms in the scene graph.
	 */
	Ogre::SceneNode *m_node;

	/*
	 * This object's unique name.
	 */
	Ogre::String m_name;


	/**
	 * Indicates the object is selected.
	 */
	bool m_isSelected;

	/**
	 * Indicates the direction this object should be moving
	 */
	Ogre::Vector3 m_direction;

	/**
	 * The initial scale of this object
	 */
	Ogre::Vector3 m_scale;

	/*
	 * Get the previous position this object was
	 */
	Ogre::Vector3 m_oldPosition;

	/**
	 * Indicates the object has been initialized so it can be used in a scene.
	 */
	bool m_isInitialized;

	int m_points;

};

#endif /* GAMEOBJECT_H_ */
