/*
 * PhysicsGameObject.cpp
 *
 *  Created on: Mar 22, 2014
 *      Author: cpresser
 */

#include "PhysicsGameObject.h"

#include <OgreBulletDynamicsRigidBody.h>
#include <Shapes/OgreBulletCollisionsBoxShape.h>
#include <Shapes/OgreBulletCollisionsSphereShape.h>
#include <Shapes/OgreBulletCollisionsCylinderShape.h>
#include <Shapes/OgreBulletCollisionsCompoundShape.h>
#include <Utils/OgreBulletCollisionsMeshToShapeConverter.h>

PhysicsGameObject::PhysicsGameObject():
	m_collisionShape(0),
	m_rigidBody(0),
	m_restitution(0.5f),
	m_friction(0.5f),
	m_mass(1.0f)
{

}

PhysicsGameObject::PhysicsGameObject(OgreBulletDynamics::DynamicsWorld* world,
		float restitution, float friction, float mass, bool dynamic,
		Ogre::SceneManager* sceneMgr, Ogre::String name, Ogre::Vector3 position,
		Ogre::String mesh, Ogre::String material, Ogre::Vector3 scale):
			m_collisionShape(0),
			m_rigidBody(0),
			m_restitution(restitution),
			m_friction(friction),
			m_mass(mass)
{
	init(world, restitution, friction, mass, dynamic, sceneMgr, name, position,
			mesh, material, scale);
}

PhysicsGameObject::~PhysicsGameObject() {

	if(m_rigidBody)
		delete m_rigidBody;

	if(m_collisionShape)
		delete m_collisionShape;

}

OgreBulletCollisions::CollisionShape* PhysicsGameObject::createCollisionShape() {
	return createConcaveCollisionShape();
}

OgreBulletCollisions::CollisionShape* PhysicsGameObject::createBoxCollisionShape() {
	if(!m_isInitialized)
		return 0;

	//figure out the bounding box for the entity
	Ogre::AxisAlignedBox bound = this->getTransformedBoundingBox();//m_entity->getWorldBoundingBox();
	Ogre::Vector3 desiredSize = (bound.getMaximum() - bound.getMinimum())*m_scale;
	return new OgreBulletCollisions::BoxCollisionShape(desiredSize);
}

OgreBulletCollisions::CollisionShape* PhysicsGameObject::createSphereCollisionShape() {
	if(!m_isInitialized)
		return 0;

	//figure out the bounding box for the entity
	Ogre::Sphere bound = m_entity->getWorldBoundingSphere();
	Ogre::Real max = m_scale.x;
	if(max < m_scale.y)
		max = m_scale.y;
	if(max < m_scale.z)
		max = m_scale.z;

	return new OgreBulletCollisions::SphereCollisionShape(max*bound.getRadius());

	//OgreBulletCollisions::StaticMeshToShapeConverter makeShape(m_entity);
	//return makeShape.createSphere();
}


OgreBulletCollisions::CollisionShape* PhysicsGameObject::createCylinderCollisionShape() {
	if(!m_isInitialized)
		return 0;

	//figure out the bounding box for the entity
	//Ogre::AxisAlignedBox bound = m_entity->getWorldBoundingBox();
	//Ogre::Vector3 desiredSize = bound.getSize();


	Ogre::Sphere bound = m_entity->getWorldBoundingSphere();
	Ogre::Vector3 desiredSize = m_scale*bound.getRadius()*Ogre::Vector3::UNIT_SCALE;
	return new OgreBulletCollisions::CylinderCollisionShape(desiredSize,
			Ogre::Vector3::UNIT_Y);

/*
	Ogre::Matrix4 transform;
	transform.setScale(m_scale);
	OgreBulletCollisions::StaticMeshToShapeConverter makeShape(m_entity, transform);
	OgreBulletCollisions::CollisionShape *result = makeShape.createCylinder();
	return result;
	*/

}

OgreBulletCollisions::CollisionShape* PhysicsGameObject::createConcaveCollisionShape() {
	if(!m_isInitialized)
		return 0;

	Ogre::Matrix4 transform = Ogre::Matrix4::IDENTITY;
	transform.setScale(m_scale);
	OgreBulletCollisions::StaticMeshToShapeConverter makeShape(m_entity, transform);
	return makeShape.createConvexDecomposition();


}

void PhysicsGameObject::init(OgreBulletDynamics::DynamicsWorld* world,
		float restitution, float friction, float mass, bool dynamic,
		Ogre::SceneManager* sceneMgr, Ogre::String name, Ogre::Vector3 position,
		Ogre::String mesh, Ogre::String material, Ogre::Vector3 scale) {

	GameObject::init(sceneMgr, name, position, mesh, material, scale);
	m_restitution = restitution;
	m_friction = friction;
	m_mass = mass;

	//everything else is initialized, set up the physics
	m_collisionShape = createCollisionShape();
	m_rigidBody = new OgreBulletDynamics::RigidBody(name + "_BODY", world);

	if(dynamic){
		m_rigidBody->setShape(m_node, m_collisionShape,
				m_restitution, m_friction, m_mass, position);
	}
	else {
		m_rigidBody->setStaticShape(m_collisionShape,
				m_restitution, m_friction, position);
	}
	m_rigidBody->setDamping(0.5, 0.5); //linear and rotational
}

const btCollisionObject* PhysicsGameObject::getBulletCollisionObject() {
	return m_rigidBody->getBulletRigidBody();
}
