/*
 * PizzaShooterApp.cpp
 *
 *  Created on: Mar 18, 2014
 *      Author: cpresser
 */

//REMEMBER: NULL = 0 = false
//      Non-NULL = not 0 = true

#include "PizzaShooterApp.h"
#include "PhysicsGameObject.h"
#include "PlaneGOB.h"
#include "PizzaGOB.h"
#include "OgreGOB.h"
#include "DirectionPointer.h"
#include "TargetCubeGOB.h"

#include <OgreBulletDynamicsWorld.h>


//#include <Shapes/OgreBulletCollisionsStaticPlaneShape.h>
//#include <Shapes/OgreBulletCollisionsSphereShape.h>

const Ogre::Vector3 PizzaShooterApp::DEFAULT_CAMERA_LOCATION = Ogre::Vector3(0, 30, 100);
const Ogre::Vector3 PizzaShooterApp::DEFAULT_CAMERA_LOOK_AT = Ogre::Vector3(0, 0, -50);

PizzaShooterApp::PizzaShooterApp():
	m_objectMap(),
	m_selectedObject(0),
	m_rayQuery(0),
	m_gravity(0, -98.1, 0),
	m_bounds(),
	m_world(0),
	m_debugDrawer(0),
	m_shotFired(false),
	m_numTargets(0),
	m_resultPanel(0),
	m_score(0)
{

}

PizzaShooterApp::~PizzaShooterApp() {
	//clean up the memory you created
	if(m_rayQuery)
		mSceneMgr->destroyQuery(m_rayQuery);

	std::map<Ogre::String, GameObject *>::iterator itMap = m_objectMap.begin();
	while(m_objectMap.end() != itMap){
		delete (itMap->second);
		++itMap;
	}

	m_objectMap.clear();

	if(m_world)
		delete m_world;

	if(m_debugDrawer)
		delete m_debugDrawer;
}


void PizzaShooterApp::createScene() {
    //add Bullet stuff
    //init the simluation object
    m_world = new OgreBulletDynamics::DynamicsWorld(mSceneMgr, m_bounds,
    		m_gravity);
    m_debugDrawer = new OgreBulletCollisions::DebugDrawer();
    //look at the "setDraw methods"
    m_debugDrawer->setDrawWireframe(true);

    m_world->setDebugDrawer(m_debugDrawer);
    m_world->setShowDebugShapes(true);

    mSceneMgr->setShadowTechnique(Ogre::SHADOWTYPE_STENCIL_ADDITIVE);
    //mSceneMgr->setShadowTechnique(Ogre::SHADOWTYPE_STENCIL_MODULATIVE);

    // Position it in Z direction
    mCamera->setPosition(DEFAULT_CAMERA_LOCATION);
    // Look back along -Z
    mCamera->lookAt(DEFAULT_CAMERA_LOOK_AT);
	//set up lights
	//a dark light (?) all around?
	mSceneMgr->setAmbientLight(Ogre::ColourValue(0.25, 0.25, 0.25));

	//a point light
	Ogre::Light *light = mSceneMgr->createLight("Light1");
	light->setType(Ogre::Light::LT_POINT);
	light->setPosition(Ogre::Vector3(0, 150, 250));
	light->setDiffuseColour(Ogre::ColourValue::White);
	light->setSpecularColour(Ogre::ColourValue::White);


	//create a sky
    mSceneMgr->setSkyBox(true, "Examples/SpaceSkyBox", 100);
    //mSceneMgr->setSkyDome(true, "Examples/CloudySky", 5, 8);

	//set up the floor
    PlaneGOB *floor = new PlaneGOB(Ogre::Vector3::UNIT_Y, 100, 100,
    		m_world, 0.5f, 0.0f, mSceneMgr, "scene_floor", Ogre::Vector3::ZERO,
    		"CPresser/Blue");

    m_objectMap["scene_floor"] = floor;


    //left wall
    PlaneGOB *leftWall1= new PlaneGOB(Ogre::Vector3::UNIT_X, 20, 100,
    		m_world, 1.0f, 0.0f, mSceneMgr, "scene_leftWall1",
    		Ogre::Vector3(-50, 10, 0),
    		"Examples/Rockwall");

    m_objectMap["scene_leftWall1"] = leftWall1;

    //right wall
    PlaneGOB *rightWall1= new PlaneGOB(Ogre::Vector3::NEGATIVE_UNIT_X, 20, 100,
    		m_world, 1.0f, 0.0f, mSceneMgr, "scene_rightWall1",
    		Ogre::Vector3(50, 10, 0),
    		"Examples/Rockwall");

    m_objectMap["scene_rightWall1"] = rightWall1;


    //back wall
    PlaneGOB *backWall1= new PlaneGOB(Ogre::Vector3::UNIT_Z, 100, 20,
    		m_world, 1.0f, 0.0f, mSceneMgr, "scene_backWall1",
    		Ogre::Vector3(0, 10, -50),
    		"Examples/Rockwall");

    m_objectMap["scene_backWall1"] = backWall1;

/*
    PhysicsGameObject *obj1 = new PhysicsGameObject(m_world, 0.5f, 0.5f, 1.0f, true,
    		mSceneMgr, "Object1", Ogre::Vector3(0, 10, -20),
       		"ellipsoid.mesh", "Examples/Chrome");

    m_objectMap["Object1"] = obj1;
*/


    //Obstacle
    PhysicsGameObject *obj2 = new PhysicsGameObject(m_world, 0.5f, 0.5f, 1.0f, false,
	mSceneMgr, "scene_wall", Ogre::Vector3(0, 1.5, -20),
		"Cube.mesh", "CPresser/SemiClearGreen", Ogre::Vector3(30, 2, 1));
		m_objectMap["scene_wall"] = obj2;

/*
    PhysicsGameObject *pizza = new PhysicsGameObject(m_world, 0.5f, 0.5f, 1.0f, true,
    		mSceneMgr, "Pizza", Ogre::Vector3(0, 0, 40), "column.mesh",
    		"CPresser/SemiClearYellow", Ogre::Vector3(0.1, 0.001, 0.1));
*/
    PizzaGOB *pizza = new PizzaGOB(m_world, mSceneMgr, "Pizza", Ogre::Vector3(0, 1, 45));
    m_objectMap["Pizza"] = pizza;

    DirectionPointer *ptr = new DirectionPointer(-Ogre::Math::HALF_PI,
    		Ogre::Math::HALF_PI, Ogre::Math::HALF_PI,
    		mSceneMgr, "Pointer", Ogre::Vector3(0, 5, 49));

    m_objectMap["Pointer"] = ptr;

    OgreGOB *ogre = new OgreGOB(m_world, mSceneMgr, "Ogre",
    		Ogre::Vector3(0, 10, -35));
    m_objectMap["Ogre"] = ogre;

    createStack(Ogre::Vector3(25, 0, 25));
    createStack(Ogre::Vector3(-25, 0, 25));
    createStack(Ogre::Vector3(25, 0, -25));
    createStack(Ogre::Vector3(-25, 0, -25));

}

void PizzaShooterApp::createFrameListener() {
	//call the base class's createFrameListner
	BaseApplication::createFrameListener();

	//remove Ogre GUI elements
	mTrayMgr->hideFrameStats();
	mTrayMgr->hideLogo();
	mTrayMgr->hideCursor();


	//set up the RaySceneQuery
	m_rayQuery = mSceneMgr->createRayQuery(Ogre::Ray());

	//make the query return the nearest one object
	m_rayQuery->setSortByDistance(true, 1);

	//set up a score board
	if (mTrayMgr->getWidget("Score")==0) {

			Ogre::StringVector items2;

			items2.push_back("Score");

			m_resultPanel = mTrayMgr->createParamsPanel(OgreBites::TL_NONE,
					"result", 200, items2);

			mTrayMgr->moveWidgetToTray(m_resultPanel, OgreBites::TL_TOPLEFT, 0);

			m_resultPanel->setParamValue(0, Ogre::StringConverter::toString(m_score));

			m_resultPanel->show();

		}



}
bool PizzaShooterApp::frameEnded(const Ogre::FrameEvent & evt)
{
	bool result = BaseApplication::frameEnded(evt);

	//updates when rendering is complete
	//m_world->stepSimulation(evt.timeSinceLastEvent);

	return result;
}



bool PizzaShooterApp::frameStarted(const Ogre::FrameEvent & evt)
{
	bool result = BaseApplication::frameStarted(evt);

	//updates prior to rendering
	//m_world->stepSimulation(evt.timeSinceLastEvent);

	return result;
}

bool PizzaShooterApp::frameRenderingQueued(const Ogre::FrameEvent& evt) {
	if(!BaseApplication::frameRenderingQueued(evt))
		return false;

	m_world->stepSimulation(evt.timeSinceLastFrame);

	//check for collisions
	manageCollisions();


	//check if we are waiting for a shot to be completed
	if(m_shotFired){
		//order the pizza
		PizzaGOB *pizza = (static_cast<PizzaGOB *>(m_objectMap["Pizza"]));
		//check if the pizza is done moving
		if(!(pizza->isMoving())){
			m_shotFired = false;

			//get a new pizza!
			delete pizza;
			pizza = new PizzaGOB(m_world, mSceneMgr, "Pizza", Ogre::Vector3(0, 1, 45));
		    m_objectMap["Pizza"] = pizza;

		    OgreGOB *ogre = static_cast<OgreGOB *>(m_objectMap["Ogre"]);
		    					ogre->stopDancing();
		}
	}

	//update all objects
	std::map<Ogre::String, GameObject*>::iterator it;
	for(it = m_objectMap.begin(); it != m_objectMap.end(); it++){
		(*it).second->update(evt);
	}


	return true;
}

bool PizzaShooterApp::keyPressed(const OIS::KeyEvent& evt) {
	switch (evt.key) {
	case OIS::KC_LEFT:
		(static_cast<DirectionPointer *>(m_objectMap["Pointer"]))->moveLeft();
		break;
	case OIS::KC_RIGHT:
		(static_cast<DirectionPointer *>(m_objectMap["Pointer"]))->moveRight();
		break;
	case OIS::KC_SPACE:
			firePizza();
		break;
	case OIS::KC_5:
	    // Position it in Z direction
	    mCamera->setPosition(DEFAULT_CAMERA_LOCATION);
	    // Look back along -Z
	    mCamera->lookAt(DEFAULT_CAMERA_LOOK_AT);
		break;
	case OIS::KC_1:
	case OIS::KC_C:
		m_world->setShowDebugShapes(!(m_world->getShowDebugShapes()));
		break;
	default:
		BaseApplication::keyPressed(evt);
		break;
	}
    //mCameraMan->injectKeyDown(evt);
	return true;
}

bool PizzaShooterApp::keyReleased(const OIS::KeyEvent& evt) {

	switch (evt.key){
	case OIS::KC_LEFT:
	case OIS::KC_RIGHT:
		(static_cast<DirectionPointer *>(m_objectMap["Pointer"]))->stopMoving();
		break;
	default:
		BaseApplication::keyReleased(evt);
		break;
	}
	return true;
}

bool PizzaShooterApp::mouseMoved(const OIS::MouseEvent& evt) {
	if (evt.state.buttonDown(OIS::MB_Right))
	{
		//let the camera manager handle mouse movements when the right button is down
	    mCameraMan->injectMouseMove(evt);
	}
	mTrayMgr->injectMouseMove(evt);
	return true;
}

bool PizzaShooterApp::mousePressed(const OIS::MouseEvent& evt,
		OIS::MouseButtonID id) {
	if (evt.state.buttonDown(OIS::MB_Right)){
		mCameraMan->injectMouseDown(evt, id);
	}
	mTrayMgr->injectMouseDown(evt, id);
	return true;
}

bool PizzaShooterApp::mouseReleased(const OIS::MouseEvent& evt,
		OIS::MouseButtonID id) {
	if (evt.state.buttonDown(OIS::MB_Right))
	{
		mCameraMan->injectMouseUp(evt, id);
	}

    mTrayMgr->injectMouseUp(evt, id);
	return true;
}

void PizzaShooterApp::firePizza(){
	if(m_shotFired)
		return;

	m_shotFired = true;

	DirectionPointer *ptr = (static_cast<DirectionPointer *>(m_objectMap["Pointer"]));
	PizzaGOB *pizza = (static_cast<PizzaGOB *>(m_objectMap["Pizza"]));

	pizza->firePizza(ptr->getCurrentValue());


}

void PizzaShooterApp::manageCollisions(){

	//have to dig into bullet a bit here
	//check the collisions
	//get the bullet world:
	btCollisionWorld *world = m_world->getBulletCollisionWorld();

	//bullet stores a cache of all of the last collisions
	int numManifolds = world->getDispatcher()->getNumManifolds();

	//iterate through the collisions
	for (int i=0;i<numManifolds;i++)
	{
		//get the bullet collision shapes
		btPersistentManifold* contactManifold =  world->getDispatcher()->getManifoldByIndexInternal(i);
		const btCollisionObject* obA = static_cast<const btCollisionObject*>(contactManifold->getBody0());
		const btCollisionObject* obB = static_cast<const btCollisionObject*>(contactManifold->getBody1());

		PhysicsGameObject *pg1 = getObjectFromBulletCollision(obA);
		PhysicsGameObject *pg2 = getObjectFromBulletCollision(obB);
		PhysicsGameObject *hitObject = NULL;

		//is one a pizza?
		if(pg1 && pg2){
			Ogre::String hitName;
			if(pg1->getName().find("Pizza") == 0 && pg2->getName().find("scene_") != 0){

				hitObject = pg2;
			}
			else if(pg2->getName().find("Pizza") == 0 && pg1->getName().find("scene_") != 0){

				hitObject = pg1;

			}
			if(hitObject){
				//score it
				m_score += hitObject->getPoints();
				m_resultPanel->setParamValue(0, Ogre::StringConverter::toString(m_score));

				if(hitObject->getName() != "Ogre"){
					//destroy it?
					m_objectMap.erase(hitObject->getName());
					delete hitObject;
				}
				else {
					OgreGOB *ogre = static_cast<OgreGOB *>(m_objectMap["Ogre"]);
					ogre->dance();
				}
			}
		}

	}

}


PhysicsGameObject *PizzaShooterApp::getObjectFromBulletCollision(const btCollisionObject *obj){
	//update all objects
	std::map<Ogre::String, GameObject*>::iterator it;
	for(it = m_objectMap.begin(); it != m_objectMap.end(); it++){

		//check if the rigid body belongs to this object
		PhysicsGameObject *pgob = dynamic_cast<PhysicsGameObject *>((*it).second);
		if(pgob){
			//not null so it must be a  PhysicsGameObject
			if(pgob->getBulletCollisionObject() == obj){
				return pgob;
			}
		}
		else {
			//must be something else

		}
	}
	return NULL;
 }

void PizzaShooterApp::createStack(Ogre::Vector3 position) {
	Ogre::String name = "CUBE_" + Ogre::StringConverter::toString(m_numTargets++);
	TargetCubeGOB *cube1 = new TargetCubeGOB(m_world, mSceneMgr, name,
			"CPresser/Green", position + 2.5*Ogre::Vector3::UNIT_Y, 2*Ogre::Vector3::UNIT_SCALE, 10);
	m_objectMap[name] = cube1;


	name = "CUBE_" + Ogre::StringConverter::toString(m_numTargets++);
	cube1 = new TargetCubeGOB(m_world, mSceneMgr, name,
			"CPresser/Yellow", position + 6.5*Ogre::Vector3::UNIT_Y, 2*Ogre::Vector3::UNIT_SCALE, 10);
	m_objectMap[name] = cube1;

	name = "CUBE_" + Ogre::StringConverter::toString(m_numTargets++);
	cube1 = new TargetCubeGOB(m_world, mSceneMgr, name,
			"CPresser/Red", position + 10.5*Ogre::Vector3::UNIT_Y, 2*Ogre::Vector3::UNIT_SCALE, 10);
	m_objectMap[name] = cube1;

}


